
import { ThemeClass } from '/sd:theme-class.js'; 
import { BasicContainer } from '/sd:basic-container.js'; 
import { StandardLabel } from '/sd:standard-label.js'; 
import { Point } from '/sd:point.js';

/**
* Class to create object instance of basic input field. It creates basic input for element.
*/
export class BasicInput extends BasicContainer {	
	/**
	* Constructor of the BasicContainer
	* @param    {jQuery} xmlObject    		data loaded from XML
	* @param    {String} parentId    		HTML ID of the object, which has to be a parent of constructing object
	* @param    {String} fieldMainClass 	Name of the CSS style's class
	* @param    {Integer} itemNumber    	Number of object constructing item - if it is null, it means, that item is not numerable
	* @param    {String} objId    			ID of creating object - if itemNumber is not null, then itemNumber will be suffix of the ID
	* @param    {String} type    			Type of input object (text, time, select)
	* @param    {String} inputId    		ID of input object
	* @param    {String} inputClass    		CSS styles class name of input field
	* @param    {Array} inputOptionsValues  If the input type is "select", then there is needed to pass Array of values for all options
	* @param    {Array} inputOptionsLabels  If the input type is "select", then there is needed to pass Array of labels for all options
	* @param    {String} labelId  			ID of label object printed above input field - if null, then label isn't printed
	* @param    {String} labelText  		Text of label object printed above input field
	* @param    {String} labelClass  		CSS styles class name of label object printed above input field
	* @param    {String} floatSeparator  	Float separator type
	* @param    {Integer} clockMode  		Clock mode (12 or 24 hours)
	*/
	constructor(xmlObject, parentId, fieldMainClass = "BasicContainer", itemNumber = null, objId = null, type = "text", inputId, inputClass="dialogWindowInput", inputOptionsValues=null, inputOptionsLabels=null, labelId=null, labelText=null, labelClass=null, floatSeparator=null, clockMode=null) {
		super(xmlObject, parentId, fieldMainClass, itemNumber, objId);
		if(this.itemNumber == null)
			this.itemNumber = 0;
		if(type != null)
			this.type = type;
		else {
			for(var i=0; i < xmlObject[this.itemNumber].children.length; i++) {
				if(xmlObject[this.itemNumber].children[i].localName == "value") {
					this.type = (xmlObject[this.itemNumber].children[i].attributes['type'].value != undefined) ? xmlObject[this.itemNumber].children[i].attributes['type'].value : "text";
				}
			}
		}
		this.floatSeparator = floatSeparator;
		this.clockMode = clockMode;
		if(labelText != null) {
			this.childLabel = new StandardLabel(null, this.id, labelClass, null, labelId, true);
			this.childLabel.setTextValue(labelText + ": ");
		}
		else
			this.childLabel = null;
		var stylesArray = window.theme.getStyleArray("dialogWindowBasicInput");
		if(this.type == "select") {
			this.childSelect = document.createElement("select");
			this.childSelect.setAttribute("id", inputId);
			this.childSelect.setAttribute("class", inputClass);
			ThemeClass.setTableAsStyle(this.childSelect, stylesArray);
			this.mainObj.appendChild(this.childSelect);
			this.childOption = new Array();
			var objPointer = null;
			for(var i = 0; i < inputOptionsValues.length; i++) {
				objPointer = document.createElement("option");
				objPointer.setAttribute("id", inputId + String(inputOptionsValues[i]).charAt(0).toUpperCase() + String(inputOptionsValues[i]).slice(1));
				objPointer.setAttribute("value", inputOptionsValues[i]);
				objPointer.setAttribute("style", "font-size:200%;");
				objPointer.innerHTML = inputOptionsLabels[i];
				this.childOption.push(objPointer);
				this.childSelect.appendChild(this.childOption[i]);
			}
		}
		else {
			this.childSelect = null;
		}
		if(this.type == "text" || this.type == "time") {
			this.childInput = document.createElement("input");
			this.childInput.setAttribute("id", inputId);
			this.childInput.setAttribute("type", this.type);
			this.childInput.setAttribute("class", inputClass);
			ThemeClass.setTableAsStyle(this.childInput, stylesArray);
			this.mainObj.appendChild(this.childInput);
		}
		else {
			this.childInput = null;
		}
		
	}

	/**
	 * Gets vaule of the input HTML object
	 * @returns {Object}		Current value
	 */
	getValue() {
		if(this.type == "text") {
			return this.childInput.value;
		}
		else if(this.type == "select") {
			return this.childSelect.options.selectedIndex;
		}
		else if(this.type == "time") {
			var value = this.childInput.value;
			return (value != null && value != undefined) ? BasicContainer.convertStringTimeToInteger(this.clockMode, value) : 0;
		}
		else {
			console.log("Not supported type: " + String(this.type));
			return null;
		}
	}

	/**
	 * Sets enabling of the input HTML object
	 * @param {Boolean} value 		Enabling value
	 * @returns {Boolean}			true if enabling was changed proper; false if something went wrong
	 */
	setEnabling(value) {
		if(this.type == "text" || this.type == "time") {
			this.childInput.disabled = (!value);
		}
		else if(this.type == "select") {
			this.childSelect.disabled = (!value);
		}
		else {
			console.log("Not supported type: " + String(this.type));
			return false;
		}
		return true;
	}

	/**
	 * Sets new value to input HTML Object
	 * @param {Object} value 		New value
	 */
	setValue(value) {
		if(this.type == "text") {
			if(this.childInput.hasAttribute("value"))
				this.childInput.value = Point.useChosenSeparator(String(value), this.floatSeparator);
			else 
				this.childInput.setAttribute("value", Point.useChosenSeparator(String(value), this.floatSeparator));
		}
		else if(this.type == "select") {
			this.childSelect.options.selectedIndex = value;
		}
		else if(this.type == "time") {
			if(this.childInput.hasAttribute("value"))
				this.childInput.value = BasicContainer.convertIntTimeToString(value, null, 24); //clock mode is fixed to 24-hours, because HTML input field works (in background) with 24h only, hours showing in HTML input depends on locale, and WebBrowser
			else 
				this.childInput.setAttribute("value", BasicContainer.convertIntTimeToString(value, null, 24)); //clock mode is fixed to 24-hours, because HTML input field works (in background) with 24h only, hours showing in HTML input depends on locale, and WebBrowser
		}
		else
			console.log("Not supported type: " + String(this.type));
	}

	/**
	 * Cahnges visibility of input HTML object
	 * @param {Boolean} value 		True if input has to be visible; false for invisibility
	 */
	setVisibility(value) {
		this.setStyleValue("display", (value == true) ? "block" : "none");
	}
	
}